// DOM Elements
const elements = {
    upload1: document.getElementById('upload1'),
    upload2: document.getElementById('upload2'),
    uploadButton: document.getElementById('uploadButton'),
    deleteButton: document.getElementById('deleteButton'),
    progressBar: document.getElementById('progressBar'),
    timestamp: document.getElementById('timestamp'),
    lastUpdated: document.getElementById('lastUpdated')
};

// State Management
let appState = {
    data1: null,
    data2: null,
    isUploading: false,
    chunkSize: 500
};

// Event Listeners
elements.upload1.addEventListener('change', checkFiles);
elements.upload2.addEventListener('change', checkFiles);
elements.uploadButton.addEventListener('click', handleFiles);
elements.deleteButton.addEventListener('click', deleteAllData);

// Initialize Application
window.addEventListener('load', () => {
    updateLastUpdatedTimestamp();
    elements.progressBar.style.display = 'none';
});

// Fetch and display the last update timestamp
async function updateLastUpdatedTimestamp() {
    try {
        const response = await fetch('last_updated.php');
        const text = await response.text();
        console.log('Server response:', text);

        const result = JSON.parse(text);
        console.log('Parsed result:', result);

        if (result.status === 'success') {
            elements.timestamp.textContent = result.timestamp;
            elements.lastUpdated.textContent = `Last Updated: ${result.timestamp}`;
        } else {
            console.error('Error fetching timestamp:', result.message);
        }
    } catch (error) {
        console.error('Error parsing JSON:', error);
    }
}

// Check if both files are selected
function checkFiles() {
    const file1 = elements.upload1.files[0];
    const file2 = elements.upload2.files[0];
    elements.uploadButton.disabled = !(file1 && file2);
}

// Handle file upload and processing
async function handleFiles() {
    if (appState.isUploading) {
        alert('Upload is already in progress. Please wait.');
        return;
    }

    const file1 = elements.upload1.files[0];
    const file2 = elements.upload2.files[0];

    if (!file1 || !file2) {
        alert('Please select both files.');
        return;
    }

    appState.isUploading = true;
    elements.uploadButton.disabled = true;
    elements.progressBar.style.display = 'block';
    elements.progressBar.value = 0;

    try {
        // Process both files
        appState.data1 = await processExcelFile(file1, ['Item', 'Barcode']);
        appState.data2 = await processExcelFile(file2, ['SKU code', 'Shelf bin', 'Available qty']);

        // Merge and upload data
        const mergedData = mergeAndUploadData(appState.data1, appState.data2);
        await chunkedUpload(mergedData);

        // Update timestamp and show success
        await updateLastUpdatedTimestamp();
        alert('Upload complete!');
    } catch (error) {
        console.error('Error during file processing:', error);
        alert(`Error: ${error.message}`);
    } finally {
        resetUploadState();
    }
}

// Process Excel file and validate columns
async function processExcelFile(file, requiredColumns) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();

        reader.onload = function (event) {
            try {
                const data = new Uint8Array(event.target.result);
                const workbook = XLSX.read(data, { type: 'array' });
                const sheetName = workbook.SheetNames[0];
                const worksheet = workbook.Sheets[sheetName];
                const jsonData = XLSX.utils.sheet_to_json(worksheet);

                console.log(`Processed ${file.name} columns:`, Object.keys(jsonData[0]));

                if (!validateFileFormat(jsonData, requiredColumns)) {
                    throw new Error(`File "${file.name}" is not in the correct format. It must contain: ${requiredColumns.join(', ')}`);
                }

                resolve(jsonData);
            } catch (error) {
                reject(error);
            }
        };

        reader.onerror = () => {
            reject(new Error(`Failed to read file: ${file.name}`));
        };

        reader.readAsArrayBuffer(file);
    });
}

// Validate file format against required columns
function validateFileFormat(data, requiredColumns) {
    if (!data || data.length === 0) return false;

    const keys = Object.keys(data[0]);
    return requiredColumns.every(column =>
        keys.some(key => key.trim().toLowerCase() === column.trim().toLowerCase())
    );
}

// Merge data from both files
function mergeAndUploadData(data1, data2) {
    const itemBarcodeMap = new Map();
    const mergedData = [];

    // Populate itemBarcodeMap with data from the first file
    data1.forEach(row => {
        const item = row['Item'];
        const barcode = row['Barcode'];
        if (!itemBarcodeMap.has(item)) {
            itemBarcodeMap.set(item, {item, barcodes: [], shelfbins: []});
        }
        itemBarcodeMap.get(item).barcodes.push(barcode);
    });

    // Update itemBarcodeMap with data from the second file
    data2.forEach(row => {
        const skuCode = row['SKU code'];
        const shelfbin = row['Shelf bin'];
        const availableQty = row['Available qty'];
        if (itemBarcodeMap.has(skuCode)) {
            if (!Array.isArray(itemBarcodeMap.get(skuCode).shelfbins)) {
                itemBarcodeMap.get(skuCode).shelfbins = [];
            }
            itemBarcodeMap.get(skuCode).shelfbins.push({
                shelfbin: shelfbin,
                'Available qty': availableQty
            });
        }
    });

    // Convert the map to an array
    itemBarcodeMap.forEach(value => {
        mergedData.push(value);
    });

    console.log('Merged Data:', mergedData);
    return mergedData;
}

// Upload data in chunks
async function chunkedUpload(data) {
    const totalChunks = Math.ceil(data.length / appState.chunkSize);

    for (let chunkIndex = 0; chunkIndex < totalChunks; chunkIndex++) {
        const chunk = data.slice(chunkIndex * appState.chunkSize, (chunkIndex + 1) * appState.chunkSize);

        try {
            const response = await fetch('insert.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(chunk)
            });

            const text = await response.text();
            console.log('Chunk upload response:', text);

            const result = JSON.parse(text);
            if (result.status !== 'success') {
                throw new Error(result.message || 'Server returned unsuccessful status');
            }

            // Update progress
            const uploaded = (chunkIndex + 1) * appState.chunkSize;
            elements.progressBar.value = (uploaded / data.length) * 100;
        } catch (error) {
            throw new Error(`Upload failed at chunk ${chunkIndex + 1}: ${error.message}`);
        }
    }
}

// Reset upload state
function resetUploadState() {
    appState.isUploading = false;
    elements.uploadButton.disabled = false;
    elements.progressBar.style.display = 'none';
    elements.progressBar.value = 0;
    elements.upload1.value = '';
    elements.upload2.value = '';
}

// Delete all data
async function deleteAllData() {
    if (!confirm('Are you sure you want to delete all data?')) return;

    try {
        const response = await fetch('delete.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' }
        });

        const result = await response.json();
        if (result.status === 'success') {
            alert('All data deleted successfully.');
            await updateLastUpdatedTimestamp();
        } else {
            throw new Error(result.message);
        }
    } catch (error) {
        console.error('Error during deletion:', error);
        alert(`Error: ${error.message}`);
    }
}