document.getElementById('barcodeInput').addEventListener('keypress', function(event) {
    if (event.key === 'Enter') {
        event.preventDefault();
        fetchShelfbin();
    }
});

document.getElementById('exportButton').addEventListener('click', exportToClipboard);
document.getElementById('exportShelfbinButton').addEventListener('click', copyShelfbinToClipboard);

// Fetch and display the last update timestamp and scan counts
window.onload = function() {
    fetch('last_updated.php', {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.status === 'success') {
            document.getElementById('timestamp').textContent = data.timestamp;
            document.getElementById('lifetimeCount').textContent = data.lifetime_count;
            console.log(data.lifetime_count);
        } else {
            console.error('Error fetching data:', data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
    });
};

function updateBarcodeList(barcode) {
    const barcodeList = document.getElementById('barcodeList');
    let currentList = barcodeList.textContent.trim();
    
    // Remove spaces and split into array
    let barcodeArray = currentList.split(',').filter(Boolean);

    // Check for duplicates
    if (!barcodeArray.includes(barcode)) {
        // Add the new barcode and update the list
        barcodeArray.push(barcode);
        barcodeList.textContent = barcodeArray.join(',');
    } else {
        return false;
    }

    document.getElementById('exportBarcodeItemButton').disabled = false; // Enable the export button
    console.log('Updated barcode list:', barcodeList.textContent); // Log the updated list
    
    return true;
}

function fetchShelfbin() {
    const barcode = document.getElementById('barcodeInput').value;
    const warning = document.getElementById('warning');
    warning.textContent = '';

    if (barcode.trim() === '') {
        warning.textContent = 'Please enter a barcode.';
        return;
    }

    fetch('fetch_shelfbin.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ barcode: barcode })
    })
    .then(response => response.text()) // Get the raw text response
    .then(text => {
        console.log('Server response:', text); // Log the raw response
        try {
            const result = JSON.parse(text); // Attempt to parse JSON
            console.log('Parsed result:', result); // Log the parsed result
            if (result.status === 'success') {
                const shelfbin = result.shelfbin;
                const item = result.item;
                if (shelfbin.startsWith('T')) {
                    warning.textContent = 'There is only Shelfbin starting with T and will not be added.';
                    document.getElementById('barcodeInput').value = ''; // Clear the input
                } else if (!shelfbin || shelfbin.trim() === '') {
                    warning.textContent = 'There is no stock in GEEK+ or no shelf with available quantity. Item will not be added.';
                    document.getElementById('barcodeInput').value = ''; // Clear the input
                } else {
                    if (updateShelfbinList(shelfbin) && updateItemList(item) && updateBarcodeList(barcode) ) {                incrementScanCount();
                    }
                }
            } else {
                warning.textContent = result.message;
                document.getElementById('barcodeInput').value = ''; // Clear the input
            }
        } catch (error) {
            console.error('Error parsing JSON:', error, 'Response text:', text);
            warning.textContent = 'An error occurred. Please try again.';
            document.getElementById('barcodeInput').value = ''; // Clear the input
        }
    })
    .catch(error => {
        console.error('Error:', error);
        warning.textContent = 'An error occurred. Please try again.';
        document.getElementById('barcodeInput').value = ''; // Clear the input
    });
}

function incrementScanCount() {
    const lifetimeCountElement = document.getElementById('lifetimeCount');

    lifetimeCountElement.textContent = parseInt(lifetimeCountElement.textContent) + 1;
}

function updateShelfbinList(shelfbin) {
    const shelfbinList = document.getElementById('shelfbinList');
    const warning = document.getElementById('warning');
    let currentList = shelfbinList.textContent.trim();
    
    // Remove spaces and split into array
    let shelfbinArray = currentList.split(',').filter(Boolean);

    // Check for duplicates
    if (shelfbinArray.includes(shelfbin)) {
        console.log('Duplicate found, not adding to list:', shelfbin);
        warning.textContent = 'Duplicate shelfbin, not adding to the list.';
        document.getElementById('barcodeInput').value = ''; // Clear the input
        return false;
    }

    // Add the new shelfbin and update the list
    shelfbinArray.push(shelfbin);
    shelfbinList.textContent = shelfbinArray.join(',');

    document.getElementById('barcodeInput').value = ''; // Clear the input
    document.getElementById('exportShelfbinButton').disabled = false; // Enable the export button
    console.log('Updated shelfbin list:', shelfbinList.textContent); // Log the updated list
    
    return true;
}

function updateItemList(item) {
    const itemList = document.getElementById('itemList');
    let currentList = itemList.textContent.trim();
    
    // Remove spaces and split into array
    let itemArray = currentList.split(',').filter(Boolean);

    // Check for duplicates
    if (!itemArray.includes(item)) {
        // Add the new item and update the list
        itemArray.push(item);
        itemList.textContent = itemArray.join(',');
    } else {
        return false;
    }
    
    document.getElementById('exportButton').disabled = false; // Enable the export button
    console.log('Updated item list:', itemList.textContent); // Log the updated list
    
    return true;
}

function exportToClipboard() {
    const itemList = document.getElementById('itemList').textContent.replace(/\s/g, '');
    if (!itemList) {
        alert('No items to export.');
        return;
    }

    const itemArray = itemList.split(',').filter(Boolean);

    // Join the items into a single string separated by newlines
    const itemsToCopy = itemArray.join('\n');

    // Copy the items to the clipboard
    navigator.clipboard.writeText(itemsToCopy).then(() => {
        alert('Items copied to clipboard. You can now paste them into Excel.');
    }).catch(err => {
        console.error('Could not copy items: ', err);
    });
}

function copyShelfbinToClipboard() {
    const shelfbinList = document.getElementById('shelfbinList').textContent;
    navigator.clipboard.writeText(shelfbinList).then(() => {
        alert('Shelfbin list copied to clipboard.');
    }).catch(err => {
        console.error('Could not copy text: ', err);
    });
}

document.getElementById('exportBarcodeItemButton').addEventListener('click', exportBarcodesAndItemsToClipboard);

function exportBarcodesAndItemsToClipboard() {
    const barcodeList = document.getElementById('barcodeList').textContent.replace(/\s/g, '');
    const itemList = document.getElementById('itemList').textContent.replace(/\s/g, '');

    if (!barcodeList || !itemList) {
        alert('No barcodes or items to export.');
        return;
    }

    const barcodeArray = barcodeList.split(',').filter(Boolean);
    const itemArray = itemList.split(',').filter(Boolean);

    if (barcodeArray.length !== itemArray.length) {
        alert('Mismatch between barcodes and items. Please check the lists.');
        return;
    }

    let tableContent = '';
    for (let i = 0; i < barcodeArray.length; i++) {
        tableContent += `${barcodeArray[i]}\t${itemArray[i]}\n`;  // Tab between columns
    }

    // Copy the table content to the clipboard
    navigator.clipboard.writeText(tableContent).then(() => {
        alert('Barcodes and Items copied to clipboard. You can now paste them into Excel.');
    }).catch(err => {
        console.error('Could not copy barcodes and items: ', err);
    });
}

