<?php
ob_start();
require_once 'config.php';

// Handle CORS preflight request
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Access-Control-Max-Age: 86400");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    ob_end_clean();
    exit;
}

// Handle different HTTP methods
switch ($_SERVER['REQUEST_METHOD']) {
    case 'OPTIONS':
        http_response_code(204);
        ob_end_clean();
        exit;
    case 'POST':
        break;
    default:
        http_response_code(405);
        $response = ["error" => "Method not allowed"];
        file_put_contents('debug.log', "store_snapshot.php: Method not allowed\n", FILE_APPEND);
        ob_end_clean();
        echo json_encode($response, JSON_THROW_ON_ERROR);
        exit;
}

// Initialize response structure
$response = [
    "success" => false,
    "data" => null,
    "message" => null,
    "error" => null,
];

try {
    // Define upload directory
    $uploadDir = __DIR__ . '/uploads/screenshots/';
    if (!file_exists($uploadDir)) {
        mkdir($uploadDir, 0777, true);
        file_put_contents('debug.log', "store_snapshot.php: Created upload directory: $uploadDir\n", FILE_APPEND);
    }

    // Verify directory is writable
    if (!is_writable($uploadDir)) {
        throw new Exception("Upload directory is not writable: $uploadDir");
    }

    // Parse request data
    $requestBody = json_decode(file_get_contents("php://input"), true);
    if (!$requestBody || !is_array($requestBody) || empty($requestBody)) {
        throw new Exception('Invalid request data: Unable to parse JSON or empty array');
    }

    // Extract the first event object (assuming single page_snapshot event)
    $request = $requestBody[0];

    // Validate required fields
    if (!isset($request['unique_id']) || !isset($request['image'])) {
        throw new Exception('Invalid request data: Missing required fields');
    }

    $unique_id = $request['unique_id'];
    $imageData = $request['image'];
    $elements = $request['elements'] ?? [];
    $viewport_width = $request['viewport_width'] ?? 1200;
    $viewport_height = $request['viewport_height'] ?? 800;
    $device_info = $request['device_info'] ?? [
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'screen_width' => 1920,
        'screen_height' => 1080,
    ];

    // Validate unique_id (you can enhance this with a database check if needed)
    $validUniqueIds = [
        'd8b21ab8-1beb-4295-858d-17a26a462e03', // Example valid unique_id
        // Add more valid unique_ids as needed
    ];
    if (!in_array($unique_id, $validUniqueIds)) {
        throw new Exception("Invalid unique_id: $unique_id");
    }

    // Validate domain (via HTTP referer or origin)
    $allowedDomains = [
        'https://www.codingio.be',
        'http://localhost:3000', // Add your local development domain if needed
    ];
    $referer = $_SERVER['HTTP_REFERER'] ?? '';
    $origin = $_SERVER['HTTP_ORIGIN'] ?? '';
    $requestDomain = parse_url($referer, PHP_URL_HOST) ?: parse_url($origin, PHP_URL_HOST);
    if (!$requestDomain) {
        throw new Exception("Unable to determine request origin");
    }
    $requestOrigin = parse_url($referer, PHP_URL_SCHEME) . '://' . $requestDomain;
    if (!in_array($requestOrigin, $allowedDomains)) {
        throw new Exception("Unauthorized domain: $requestOrigin");
    }
    file_put_contents('debug.log', "store_snapshot.php: Request origin validated: $requestOrigin\n", FILE_APPEND);

    // Validate image data
    if (!preg_match('/^data:image\/png;base64,/', $imageData)) {
        throw new Exception('Invalid image data format');
    }

    // Remove the data URI scheme prefix
    $imageData = str_replace('data:image/png;base64,', '', $imageData);
    $imageBinary = base64_decode($imageData);
    if ($imageBinary === false) {
        throw new Exception('Failed to decode image data');
    }

    // Generate timestamp and filenames
    $timestamp = date('Y-m-d_H-i-s');
    $imageFilename = "{$unique_id}_{$timestamp}.png";
    $jsonFilename = "{$unique_id}_{$timestamp}.json";
    $imagePath = $uploadDir . $imageFilename;
    $jsonPath = $uploadDir . $jsonFilename;

    // Save the image file
    if (file_put_contents($imagePath, $imageBinary) === false) {
        throw new Exception("Failed to save image to: $imagePath");
    }

    // Prepare snapshot metadata
    $snapshotData = [
        'name' => '',
        'image' => "uploads/screenshots/{$imageFilename}",
        'image_data' => null,
        'viewport_width' => $viewport_width,
        'viewport_height' => $viewport_height,
        'timestamp' => date('c'),
        'unique_id' => $unique_id,
        'elements' => $elements,
        'device_info' => $device_info,
    ];

    // Save the metadata as JSON
    if (file_put_contents($jsonPath, json_encode($snapshotData, JSON_PRETTY_PRINT)) === false) {
        unlink($imagePath);
        throw new Exception("Failed to save metadata to: $jsonPath");
    }

    $response['success'] = true;
    $response['message'] = 'Snapshot stored successfully';
    $response['data'] = $snapshotData;

} catch (Exception $e) {
    error_log("store_snapshot.php: " . $e->getMessage(), 0);
    http_response_code(400);
    $response["error"] = $e->getMessage();
} catch (JsonException $e) {
    error_log("store_snapshot.php: JSON Error: " . $e->getMessage(), 0);
    http_response_code(500);
    $response["error"] = "JSON encoding failed";
}

file_put_contents('debug.log', "store_snapshot.php: Response: " . json_encode($response) . "\n", FILE_APPEND);

ob_end_clean();
echo json_encode($response, JSON_THROW_ON_ERROR);
?>