(function() {
  const script = document.createElement('script');
  script.src = 'https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js';
  script.onload = () => console.log('html2canvas library loaded successfully');
  script.onerror = () => console.error('Failed to load html2canvas library');
  document.head.appendChild(script);
})();

// Singleton to ensure startTracking runs only once per uniqueId
const trackingInstances = new Map();

function startTracking(uniqueId) {
  if (trackingInstances.has(uniqueId)) {
    console.log(`Tracking already initialized for unique_id: ${uniqueId}, skipping...`);
    return;
  }

  console.log('tracker.js initialized with unique_id:', uniqueId);

  // Session management
  const sessionKey = `tracker_session_${uniqueId}`;
  let sessionData = JSON.parse(localStorage.getItem(sessionKey) || "{}");
  const SESSION_TIMEOUT = 30 * 60 * 1000; // 30 minutes

  // Check if session is still active
  if (!sessionData.id || (Date.now() - (sessionData.lastActivity || 0)) > SESSION_TIMEOUT) {
    sessionData = {
      id: generateSessionId(),
      lastActivity: Date.now(),
      lastPageLoad: 0
    };
    localStorage.setItem(sessionKey, JSON.stringify(sessionData));
  }

  const sessionId = sessionData.id;
  const events = [];
  const interactionTimestamps = []; // Store timestamps of interactions
  let pageLoadTime = 0;
  let isFocused = true; // Assume focused on page load
  let visibleTime = 0; // Total time tab/window is visible
  let lastFocusStart = Date.now(); // Start time of current focus period
  const pageLoadTimestamp = Date.now(); // Timestamp of page load

  // Update last activity on user interaction
  const updateActivity = () => {
    sessionData.lastActivity = Date.now();
    localStorage.setItem(sessionKey, JSON.stringify(sessionData));
  };

  // Check for screenshot mode via URL query parameter
  const urlParams = new URLSearchParams(window.location.search);
  const isScreenshotMode = urlParams.get('screenshotMode') === 'true';
  console.log('Screenshot mode check - URL params:', window.location.search, 'Result:', isScreenshotMode);

  // Reference to the screenshot button
  let screenshotButton = null;

  // Add button immediately with retry logic
  function addScreenshotButton() {
    console.log('Attempting to add Take Screenshot button...');
    screenshotButton = document.createElement('button');
    screenshotButton.id = 'take-screenshot-btn'; // Add ID for easy reference
    screenshotButton.textContent = 'Take Screenshot';
    screenshotButton.style.position = 'fixed';
    screenshotButton.style.top = '10px';
    screenshotButton.style.right = '10px';
    screenshotButton.style.zIndex = '10000';
    screenshotButton.style.padding = '10px 20px';
    screenshotButton.style.backgroundColor = '#007bff';
    screenshotButton.style.color = 'white';
    screenshotButton.style.border = 'none';
    screenshotButton.style.borderRadius = '5px';
    screenshotButton.style.cursor = 'pointer';

    screenshotButton.addEventListener('click', () => {
      captureScreenshot(uniqueId, sessionId);
    });

    const appendButton = () => {
      if (document.body) {
        document.body.appendChild(screenshotButton);
        console.log('Take Screenshot button added to DOM');
      } else {
        console.warn('Document body not available yet, retrying...');
        setTimeout(appendButton, 100);
      }
    };
    appendButton();
  }

  // Create and show modal popup
  function showModal() {
    // Remove any existing modal
    const existingModal = document.getElementById('screenshot-modal');
    if (existingModal) {
      existingModal.remove();
    }

    // Create modal container
    const modal = document.createElement('div');
    modal.id = 'screenshot-modal';
    modal.style.position = 'fixed';
    modal.style.top = '0';
    modal.style.left = '0';
    modal.style.width = '100%';
    modal.style.height = '100%';
    modal.style.backgroundColor = 'rgba(0, 0, 0, 0.5)';
    modal.style.display = 'flex';
    modal.style.justifyContent = 'center';
    modal.style.alignItems = 'center';
    modal.style.zIndex = '10001';

    // Create modal content
    const modalContent = document.createElement('div');
    modalContent.style.backgroundColor = 'white';
    modalContent.style.padding = '20px';
    modalContent.style.borderRadius = '10px';
    modalContent.style.textAlign = 'center';
    modalContent.style.maxWidth = '400px';
    modalContent.style.width = '90%';

    // Modal message
    const message = document.createElement('p');
    message.textContent = 'Screenshot taken successfully!';
    message.style.marginBottom = '20px';
    modalContent.appendChild(message);

    // Return button
    const returnButton = document.createElement('button');
    returnButton.textContent = 'Return';
    returnButton.style.padding = '10px 20px';
    returnButton.style.margin = '0 10px';
    returnButton.style.backgroundColor = '#dc3545';
    returnButton.style.color = 'white';
    returnButton.style.border = 'none';
    returnButton.style.borderRadius = '5px';
    returnButton.style.cursor = 'pointer';
    returnButton.addEventListener('click', () => {
      if (window.opener && !window.opener.closed) {
        window.close(); // Close popup if opened by another window
      } else {
        const url = new URL(window.location.href);
        url.searchParams.delete('screenshotMode');
        window.location.replace(url.toString()); // Redirect main window
      }
    });
    modalContent.appendChild(returnButton);

    // Take Another button
    const anotherButton = document.createElement('button');
    anotherButton.textContent = 'Take Another';
    anotherButton.style.padding = '10px 20px';
    anotherButton.style.margin = '0 10px';
    anotherButton.style.backgroundColor = '#28a745';
    anotherButton.style.color = 'white';
    anotherButton.style.border = 'none';
    anotherButton.style.borderRadius = '5px';
    anotherButton.style.cursor = 'pointer';
    anotherButton.addEventListener('click', () => {
      modal.remove(); // Close modal
      screenshotButton.style.display = 'block'; // Show button again
    });
    modalContent.appendChild(anotherButton);

    modal.appendChild(modalContent);
    document.body.appendChild(modal);
  }

  if (isScreenshotMode) {
    console.log('Entering screenshot mode, adding button...');
    addScreenshotButton();
  } else {
    console.log('Entering tracking mode...');
    // Tracking logic (only if not in screenshot mode)
    window.addEventListener("load", () => {
      const now = Date.now();
      if (now - (sessionData.lastPageLoad || 0) > 1000) {
        pageLoadTime = calculatePageLoadTime();
        const pageLoadEvent = {
          unique_id: uniqueId,
          event_type: "page_load",
          timestamp: new Date().toISOString(),
          url: window.location.href || "unknown",
          load_time: pageLoadTime,
          referrer: document.referrer,
          user_agent: navigator.userAgent,
          session_id: sessionId,
          screen_width: window.screen.width,
          screen_height: window.screen.height
        };
        events.push(pageLoadEvent);
        sessionData.lastPageLoad = now;
        localStorage.setItem(sessionKey, JSON.stringify(sessionData));
      }

      const sessionStartEvent = {
        unique_id: uniqueId,
        event_type: "session_start",
        timestamp: new Date().toISOString(),
        url: window.location.href || "unknown",
        session_id: sessionId,
      };
      events.push(sessionStartEvent);

      if (events.length > 0) {
        sendData(events.splice(0, events.length));
      }

      window.addEventListener("focus", () => {
        isFocused = true;
        lastFocusStart = Date.now();
      });
      window.addEventListener("blur", () => {
        if (isFocused) {
          visibleTime += Date.now() - lastFocusStart;
          isFocused = false;
        }
      });

      document.addEventListener("visibilitychange", () => {
        if (document.hidden) {
          if (isFocused) {
            visibleTime += Date.now() - lastFocusStart;
            isFocused = false;
          }
        } else {
          isFocused = true;
          lastFocusStart = Date.now();
        }
      });

      document.addEventListener("click", (e) => {
        updateActivity();
        const clickEvent = {
          unique_id: uniqueId,
          event_type: "click",
          timestamp: new Date().toISOString(),
          url: window.location.href || "unknown",
          session_id: sessionId,
          click_x: (e.clientX / window.innerWidth) * 100,
          click_y: (e.clientY / window.innerHeight) * 100,
          element: {
            tag: e.target.tagName.toLowerCase(),
            id: e.target.id || "",
            class: e.target.className || "",
            text: (e.target.innerText || "").trim().substring(0, 50)
          }
        };
        events.push(clickEvent);
        if (isFocused) interactionTimestamps.push(Date.now());
      });

      function findScrollableElement() {
        const candidates = document.querySelectorAll('body, [style*="overflow"], [style*="scroll"], div, section');
        let scrollableElement = window;
        let maxScrollHeight = document.documentElement.scrollHeight;

        for (let element of candidates) {
          const style = window.getComputedStyle(element);
          const isOverflow = style.overflowY === 'auto' || style.overflowY === 'scroll';
          const elementScrollHeight = element.scrollHeight;
          const elementClientHeight = element.clientHeight;

          if (isOverflow && elementScrollHeight > elementClientHeight && elementScrollHeight > maxScrollHeight) {
            scrollableElement = element;
            maxScrollHeight = elementScrollHeight;
          }
        }

        return scrollableElement;
      }

      let maxScrollDepth = 0;
      let lastSentScrollDepth = 0;
      let lastScrollUpdate = 0;
      let isScrollable = false;
      const scrollableElement = findScrollableElement();
      console.log("Debug - Scrollable element:", scrollableElement === window ? "window" : scrollableElement);

      const scrollHandler = () => {
        const now = Date.now();
        if (now - lastScrollUpdate < 500) return;
        lastScrollUpdate = now;

        updateActivity();
        let scrollHeight, clientHeight, scrollPosition;
        if (scrollableElement === window) {
          scrollHeight = document.documentElement.scrollHeight;
          clientHeight = window.innerHeight;
          scrollPosition = window.scrollY;
        } else {
          scrollHeight = scrollableElement.scrollHeight;
          clientHeight = scrollableElement.clientHeight;
          scrollPosition = scrollableElement.scrollTop;
        }

        if (scrollHeight > clientHeight) {
          isScrollable = true;
          const scrollDepth = Math.round((scrollPosition / (scrollHeight - clientHeight)) * 100);
          maxScrollDepth = Math.max(maxScrollDepth, scrollDepth);
        }
        if (isFocused) interactionTimestamps.push(Date.now());
      };

      scrollableElement.addEventListener("scroll", scrollHandler);

      window.addEventListener("unload", () => {
        if (isFocused) {
          visibleTime += Date.now() - lastFocusStart;
        }

        let hybridTime = visibleTime;
        if (interactionTimestamps.length > 0) {
          let activeTime = 0;
          for (let i = 0; i < interactionTimestamps.length - 1; i++) {
            activeTime += interactionTimestamps[i + 1] - interactionTimestamps[i];
          }
          hybridTime += activeTime;
        }

        if (hybridTime === 0 || hybridTime < 1000) {
          const lastEventTime = Math.max(sessionData.lastActivity || Date.now(), ...interactionTimestamps);
          hybridTime = lastEventTime - pageLoadTimestamp;
        }

        if (isScrollable && maxScrollDepth > 0 && maxScrollDepth !== lastSentScrollDepth) {
          const scrollEvent = {
            unique_id: uniqueId,
            event_type: "scroll",
            timestamp: new Date().toISOString(),
            url: window.location.href || "unknown",
            session_id: sessionId,
            scroll_depth: maxScrollDepth
          };
          events.push(scrollEvent);
          lastSentScrollDepth = maxScrollDepth;
        }

        if (hybridTime > 0) {
          const timeSpentEvent = {
            unique_id: uniqueId,
            event_type: "time_spent",
            timestamp: new Date().toISOString(),
            url: window.location.href || "unknown",
            session_id: sessionId,
            duration: Math.round(hybridTime / 1000)
          };
          events.push(timeSpentEvent);
        }

        if (events.length > 0) {
          sendData(events);
        }
      });

      setInterval(() => {
        if (isFocused) {
          visibleTime += Date.now() - lastFocusStart;
          lastFocusStart = Date.now();
        }

        let hybridTime = visibleTime;
        if (interactionTimestamps.length > 0) {
          let activeTime = 0;
          for (let i = 0; i < interactionTimestamps.length - 1; i++) {
            activeTime += interactionTimestamps[i + 1] - interactionTimestamps[i];
          }
          hybridTime += activeTime;
        }

        if (hybridTime === 0 || hybridTime < 1000) {
          const lastEventTime = Math.max(sessionData.lastActivity || Date.now(), ...interactionTimestamps);
          hybridTime = lastEventTime - pageLoadTimestamp;
        }

        if (isScrollable && maxScrollDepth > 0 && maxScrollDepth !== lastSentScrollDepth) {
          const scrollEvent = {
            unique_id: uniqueId,
            event_type: "scroll",
            timestamp: new Date().toISOString(),
            url: window.location.href || "unknown",
            session_id: sessionId,
            scroll_depth: maxScrollDepth
          };
          events.push(scrollEvent);
          lastSentScrollDepth = maxScrollDepth;
        }

        if (hybridTime > 0) {
          const timeSpentEvent = {
            unique_id: uniqueId,
            event_type: "time_spent",
            timestamp: new Date().toISOString(),
            url: window.location.href || "unknown",
            session_id: sessionId,
            duration: Math.round(hybridTime / 1000)
          };
          events.push(timeSpentEvent);
        }

        if (events.length > 0) {
          sendData(events.splice(0, events.length));
        }
      }, 5000);
    });
  }

  // Function to capture screenshot
  function captureScreenshot(uniqueId, sessionId) {
    console.log('Capturing screenshot...');
    if (typeof window.html2canvas !== 'function') {
      console.error('html2canvas is not available. Ensure the library is loaded.');
      return;
    }

    // Hide the screenshot button before capturing
    if (screenshotButton) {
      screenshotButton.style.display = 'none';
      console.log('Hid Take Screenshot button for capture');
    }

    const width = window.innerWidth;
    const height = window.innerHeight;

    console.log('Starting html2canvas rendering with width:', width, 'height:', height);
    window.html2canvas(document.body, {
      width: width,
      height: height,
      logging: true,
      useCORS: true,
      allowTaint: false
    }).then(canvas => {
      console.log('html2canvas rendering completed');
      const imageBase64 = canvas.toDataURL('image/png');
      console.log('Screenshot captured, imageBase64 length:', imageBase64.length);

      // Clean URL by removing screenshotMode parameter
      let url = window.location.href;
      const urlObj = new URL(url);
      urlObj.searchParams.delete('screenshotMode');
      url = urlObj.toString();

      const screenshotEvent = {
        event_type: 'page_snapshot',
        unique_id: uniqueId,
        session_id: sessionId,
        url: url,
        image: imageBase64,
        viewport_width: width,
        viewport_height: height,
        timestamp: new Date().toISOString(),
        elements: collectElementPositions()
      };
      console.log('Prepared page snapshot event:', screenshotEvent);
      sendScreenshotData([screenshotEvent]).then(() => {
        // Show the modal after the screenshot is sent
        showModal();
      });
    }).catch(error => {
      console.error('Error during html2canvas rendering:', error);
      // Show button again even if there's an error
      if (screenshotButton) {
        screenshotButton.style.display = 'block';
      }
    });
  }

  // Function to collect element positions
  function collectElementPositions() {
    const elements = [];
    const tags = ['button', 'a', 'input', 'textarea', 'select'];
    tags.forEach(tag => {
      document.querySelectorAll(tag).forEach(el => {
        // Skip the screenshot button
        if (el.id === 'take-screenshot-btn') return;
        const rect = el.getBoundingClientRect();
        elements.push({
          tag: el.tagName.toLowerCase(),
          id: el.id || '',
          class: el.className || '',
          text: (el.innerText || el.value || '').trim().substring(0, 50),
          position: {
            x: rect.left,
            y: rect.top,
            width: rect.width,
            height: rect.height
          }
        });
      });
    });
    return elements;
  }

  function calculatePageLoadTime() {
    const performance = window.performance;
    if (performance && performance.timing) {
      const loadTime = performance.timing.domContentLoadedEventEnd - performance.timing.navigationStart;
      return loadTime >= 0 ? loadTime : 0;
    }
    return 0;
  }

  function generateSessionId() {
    return "sess_" + Date.now() + "_" + Math.random().toString(36).substr(2, 9);
  }

  function sendData(events) {
    console.log('Sending tracking events to server:', JSON.stringify(events));
    if (!events.length) {
      console.log('No tracking events to send');
      return Promise.resolve();
    }

    // Filter out non-tracking events (e.g., page_snapshot)
    const trackingEvents = events.filter(event => event.event_type !== 'page_snapshot');

    if (!trackingEvents.length) {
      console.log('No tracking events to send after filtering');
      return Promise.resolve();
    }

    return fetch("https://www.codingio.be/tweaktip/track.php", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Accept": "application/json",
      },
      body: JSON.stringify(trackingEvents),
      keepalive: true,
    })
      .then(response => {
        console.log('Server response status (track.php):', response.status);
        if (!response.ok) throw new Error(`Track fetch failed: ${response.status}`);
        return response.json();
      })
      .then(data => {
        console.log('Server response data (track.php):', data);
        return data;
      })
      .catch(error => {
        console.error('Error sending tracking data to server:', error);
        throw error;
      });
  }

  function sendScreenshotData(events) {
    console.log('Sending screenshot data to server:', JSON.stringify(events));
    if (!events.length) {
      console.log('No screenshot data to send');
      return Promise.resolve();
    }

    // Ensure only page_snapshot events are sent
    const screenshotEvents = events.filter(event => event.event_type === 'page_snapshot');

    if (!screenshotEvents.length) {
      console.log('No screenshot events to send after filtering');
      return Promise.resolve();
    }

    return fetch("https://www.codingio.be/tweaktip/store_snapshot.php", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "Accept": "application/json",
      },
      body: JSON.stringify(screenshotEvents),
      keepalive: true,
    })
      .then(response => {
        console.log('Server response status (store_snapshot.php):', response.status);
        if (!response.ok) throw new Error(`Screenshot fetch failed: ${response.status}`);
        return response.json();
      })
      .then(data => {
        console.log('Server response data (store_snapshot.php):', data);
        return data;
      })
      .catch(error => {
        console.error('Error sending screenshot data to server:', error);
        throw error;
      });
  }

  // Mark this instance as initialized
  trackingInstances.set(uniqueId, true);
}

// Initialize with unique_id from script URL or data-id attribute
const scriptElement = document.currentScript || (function() {
  const scripts = document.getElementsByTagName("script");
  return scripts[scripts.length - 1];
})();

const uniqueId = scriptElement.getAttribute("data-id");
if (uniqueId) {
  console.log("Attempting to start tracking with unique_id:", uniqueId);
  startTracking(uniqueId);
} else {
  console.error("No data-id attribute found in script tag");
}